
/* PoliceCamShake.fx
   Simulates smooth subtle vertical shake mimicking bodycam walking
   Inspired by tactical bodycam footage
   (c) 2025 by Viqus – public domain */

#include "ReShade.fxh"
#include "ReShadeUI.fxh"

// Timer source for oscillation
uniform float Timer < source = "timer"; >;

// Activate shake while holding W key (0x57)
uniform bool isWalking < source = "key"; keycode = 0x57; mode = "hold"; >;

// Shake intensity slider
uniform float ShakeIntensity <
    __UNIFORM_SLIDER_FLOAT1
    ui_label = "Shake Intensity";
    ui_tooltip = "Vertical shake amplitude";
    ui_min = 0.0; ui_max = 0.02;
> = 0.005;

// Shake frequency slider
uniform float ShakeSpeed <
    __UNIFORM_SLIDER_FLOAT1
    ui_label = "Shake Frequency";
    ui_tooltip = "Speed of the shake oscillation";
    ui_min = 0.5; ui_max = 10.0;
> = 3.5;

float3 PoliceShakePS(float4 pos : SV_Position, float2 texcoord : TEXCOORD) : SV_Target
{
    if (isWalking)
    {
        // Combined sine waves for smooth irregular motion
        float t = Timer * ShakeSpeed;
        float wave1 = sin(t * 2.0);
        float wave2 = sin(t * 3.7 + 1.3);
        float yShake = (wave1 + wave2) * 0.5 * ShakeIntensity;

        texcoord.y += yShake;
        // Prevent sampling outside the screen
        texcoord = clamp(texcoord, float2(0.0, 0.0), float2(1.0, 1.0));
    }

    return tex2D(ReShade::BackBuffer, texcoord).rgb;
}

technique PoliceCamShake
<
    ui_label = "Police BodyCam Shake";
    ui_tooltip = "Adds smooth vertical shake to simulate walking with a bodycam.";
>
{
    pass
    {
        VertexShader = PostProcessVS;
        PixelShader = PoliceShakePS;
    }
}
